import { Router } from 'express';
import { invoicesController } from './controller';
import { authenticateUser, requireVendor } from '@/middlewares/auth';
import { checkFeature } from '@/middlewares/featureGate';
import { vendorService } from '@/modules/vendors/service';

const router = Router();

// Attach vendor to req for downstream use
async function attachVendor(req: any, res: any, next: any) {
  if (req.user) {
    const { data: vendor } = await vendorService.getVendorByUserId(req.user.id);
    req.vendor = vendor || null;
  }
  next();
}

// Create invoice
router.post('/', authenticateUser, requireVendor, attachVendor, checkFeature('invoicing_enabled'), invoicesController.create);
// View invoice (public)
router.get('/view/:public_code', invoicesController.viewPublic);
// Mark invoice as paid (webhook)
router.post('/:id/mark-paid', invoicesController.markPaid);
// List invoices for vendor
router.get('/', authenticateUser, requireVendor, attachVendor, checkFeature('invoicing_enabled'), invoicesController.list);
// Download PDF
router.get('/:id/download', authenticateUser, requireVendor, attachVendor, checkFeature('invoicing_enabled'), invoicesController.downloadPdf);
// Delete invoice
router.delete('/:id', authenticateUser, requireVendor, attachVendor, checkFeature('invoicing_enabled'), invoicesController.delete);
// List transactions for vendor
router.get('/transactions', authenticateUser, requireVendor, attachVendor, checkFeature('invoicing_enabled'), invoicesController.listTransactions);

export default router; 